/**
* \file: WaylandSurface.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: Android Auto
*
* \author: J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
*
* \copyright (c) 2013 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <stdio.h>
#include <string.h>
#include <adit_logging.h>
#include "WaylandSurface.h"
#include <sys/mman.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/epoll.h>
#include <string.h>
#include <stdlib.h>

#include "os-compatibility.h"

LOG_IMPORT_CONTEXT(aauto_input)

using namespace std;

namespace adit { namespace aauto
{

WaylandSurface::WaylandSurface()
{
    context = nullptr;
    surface = nullptr;
    buffer = nullptr;
    shm_data = nullptr;
    shm_size = 0;
    surfaceId = 0;
    memset(&adapterSurfaceContext, 0, sizeof(adapterSurfaceContext));
}

bool WaylandSurface::Create(WaylandContext* inContext,
        uint32_t inWidth, uint32_t inHeight, uint32_t inLayerId, uint32_t inSurfaceId)
{
    struct wl_shm_pool *pool;
    int fd, size, stride, bpp, adapter_error;
    void *data;

    if (surface != nullptr)
    {
        LOG_ERROR((aauto_input, "WaylandSurface already created"));
        return false;
    }

    context = inContext; // hold on the context as shared_ptr until the surface gets destroyed
    surfaceId = inSurfaceId;

    if (context->GetShmFormat() == WL_SHM_FORMAT_C8 )
        bpp = 1;
    else
        bpp = 4;

    stride = inWidth*bpp;
    size = stride*inHeight;

    surface = wl_compositor_create_surface(context->GetCompositor());
    if (surface == nullptr)
    {
        LOG_ERROR((aauto_input, "wl_compositor_create_surface failed"));
        return false;
    }

    fd = os_create_anonymous_file(size);
    if (fd < 0) {
        LOG_ERROR((aauto_input, "creating a buffer file for %d B failed: %m", size));
        return false;
    }

    /*pointer to the buffer memory which will be displayed*/
    data = mmap(NULL, size, PROT_READ | PROT_WRITE, MAP_SHARED, fd, 0);
    if (data == MAP_FAILED) {
        LOG_ERROR((aauto_input, "mmap failed: %m"));
        close(fd);
        return false;
    }

    pool = wl_shm_create_pool(context->GetShm(), fd, size);
    buffer = wl_shm_pool_create_buffer(pool, 0, inWidth, inHeight,
                                       stride, context->GetShmFormat());

    wl_shm_pool_destroy(pool);
    close(fd);

    memset(data, 1, size);

    shm_data = data;
    shm_size = size;

    adapter_error = compositor_shim_surface_init (&adapterSurfaceContext, surface, inLayerId,
                                                  surfaceId, inWidth, inHeight);

    if (adapter_error != 0)
    {
        LOG_ERROR((aauto_input, "compositor_shim_surface_init failed"));
        return false;
    }

    adapter_error = compositor_shim_surface_configure(context->GetAdapterContext(),&adapterSurfaceContext,
                                                      ADAPTER_CONFIGURATION_ALL);
    if (adapter_error != 0)
    {
        LOG_ERROR((aauto_input, "compositor_shim_surface_init failed"));
        return false;
    }

    wl_surface_attach(surface, buffer, 0, 0);
    wl_surface_damage(surface, 0, 0, inWidth, inHeight);
    wl_surface_commit(surface);
    wl_display_flush(context->GetDisplay());

    LOGD_DEBUG((aauto_input, "touch surface created"));
    return true;
}

WaylandSurface::~WaylandSurface()
{
    LOGD_DEBUG((aauto_input, "destroy touch surface"));

    if (surfaceId != 0)
        compositor_shim_surface_destroy(&adapterSurfaceContext);

    if (buffer != nullptr)
        wl_buffer_destroy(buffer);

    if (surface != nullptr)
        wl_surface_destroy(surface);

    if (shm_data != nullptr)
        munmap(shm_data, shm_size);
}

} } // namespace adit { namespace aauto

